package com.gc.nfc.ui;

import android.content.Intent;
import android.net.Uri;
import android.nfc.NdefMessage;
import android.nfc.NdefRecord;
import android.nfc.NfcAdapter;
import android.nfc.Tag;
import android.nfc.tech.Ndef;
import android.os.Bundle;
import android.os.Parcelable;
import android.text.TextUtils;
import android.util.Log;
import android.widget.TextView;

import com.gc.nfc.R;
import com.gc.nfc.base.BaseNfcActivity;
import com.gc.nfc.UriPrefix;

import java.nio.charset.Charset;
import java.util.Arrays;

public class ReadUriActivity extends BaseNfcActivity {
    private TextView mNfcText;
    private String mTagText;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_read_uri);
        mNfcText = (TextView) findViewById(R.id.tv_nfctext);
    }

    @Override
    public void onNewIntent(Intent intent) {
        //Get tag object
        Tag detectedTag = intent.getParcelableExtra(NfcAdapter.EXTRA_TAG);
        //Get Ndef object
        Ndef ndef = Ndef.get(detectedTag);
        if (ndef == null) return;
        mTagText = ndef.getType() + "\n max size:" + ndef.getMaxSize() + " bytes\n\n";
        readNfcTag(intent);
        mNfcText.setText(mTagText);
    }

    /**
     * Read NFC tag URI
     */
    private void readNfcTag(Intent intent) {
        if (NfcAdapter.ACTION_NDEF_DISCOVERED.equals(intent.getAction())) {
            Parcelable[] rawMsgs = intent.getParcelableArrayExtra(
                    NfcAdapter.EXTRA_NDEF_MESSAGES);
            NdefMessage ndefMessage = null;
            int contentSize = 0;
            if (rawMsgs != null) {
                if (rawMsgs.length > 0) {
                    ndefMessage = (NdefMessage) rawMsgs[0];
                    contentSize = ndefMessage.toByteArray().length;
                } else {
                    return;
                }
            }
            try {
                NdefRecord ndefRecord = ndefMessage.getRecords()[0];
                Log.i("JAVA", ndefRecord.toString());
                Uri uri = parse(ndefRecord);
                if (uri != null && !TextUtils.isEmpty(uri.toString())) {
                    contentSize = uri.toString().length();
                }
                Log.i("JAVA", "uri:" + uri.toString());
                mTagText += this.getResources().getString(R.string.will_read_uri) + uri.toString() + "\n\nUri\n" + contentSize + " bytes";
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Analysis of URI data in ndefrecord
     *
     * @param record
     * @return
     */
    public static Uri parse(NdefRecord record) {
        short tnf = record.getTnf();
        if (tnf == NdefRecord.TNF_WELL_KNOWN) {
            return parseWellKnown(record);
        } else if (tnf == NdefRecord.TNF_ABSOLUTE_URI) {
            return parseAbsolute(record);
        }
        throw new IllegalArgumentException("Unknown TNF " + tnf);
    }

    /**
     * There is no URI identifier to handle absolute URI, that is, there is no URI prefix, and all stored are strings
     *
     * @param ndefRecord An information segment describing nDef information. An ndefmessage may contain one or more ndefrds
     * @return
     */
    private static Uri parseAbsolute(NdefRecord ndefRecord) {
        //Get all the byte data
        byte[] payload = ndefRecord.getPayload();
        Uri uri = Uri.parse(new String(payload, Charset.forName("UTF-8")));
        return uri;
    }

    /**
     * Process URI of known type
     *
     * @param ndefRecord
     * @return
     */
    private static Uri parseWellKnown(NdefRecord ndefRecord) {
        //Determine whether the data is URI type
        if (!Arrays.equals(ndefRecord.getType(), NdefRecord.RTD_URI))
            return null;
        //Get all the byte data
        byte[] payload = ndefRecord.getPayload();
        String prefix = UriPrefix.URI_PREFIX_MAP.get(payload[0]);
        byte[] prefixBytes = prefix.getBytes(Charset.forName("UTF-8"));
        byte[] fullUri = new byte[prefixBytes.length + payload.length - 1];
        System.arraycopy(prefixBytes, 0, fullUri, 0, prefixBytes.length);
        System.arraycopy(payload, 1, fullUri, prefixBytes.length, payload.length - 1);
        Uri uri = Uri.parse(new String(fullUri, Charset.forName("UTF-8")));
        return uri;
    }
}
